<?php
/******************************************************
 * Push Confirmed Income from MySQL (qrcode) to Odoo
 * - يقرأ فواتير incomtype='confirmed' ضمن تاريخ (من/إلى)
 * - يحسب الإجماليات مثل تقريرك تمامًا
 * - يرسل JSON إلى أودو على دفعات: partners + invoices
 ******************************************************/

mb_internal_encoding('UTF-8');
date_default_timezone_set('Asia/Riyadh');
error_reporting(E_ALL);
ini_set('display_errors', 1);

require __DIR__ . '/conn.php'; // يُعرّف $conn كـ mysqli
if (method_exists($conn, 'set_charset')) $conn->set_charset('utf8mb4');

/* ================== الإعدادات ================== */
$ODOO_ENDPOINT = 'https://YOUR-ODOO-DOMAIN/ace/api/bulk_import'; // عدّل
$ODOO_TOKEN    = 'PUT-A-LONG-RANDOM-SECRET';                      // عدّل
$DEFAULT_VAT_RATE = 0.15; // 15%
$BATCH_SIZE   = isset($_GET['batch']) ? max(50, (int)$_GET['batch']) : 500;

/* ========== تواريخ الإدخال ========== */
$from_date = isset($_GET['from_date']) ? trim($_GET['from_date']) : date('Y-m-d');
$to_date   = isset($_GET['to_date'])   ? trim($_GET['to_date'])   : date('Y-m-d');

$re = '/^\d{4}-\d{2}-\d{2}$/';
if (!preg_match($re, $from_date)) $from_date = date('Y-m-d');
if (!preg_match($re, $to_date))   $to_date   = $from_date;

$fromObj = new DateTime($from_date);
$toObj   = new DateTime($to_date);
if ($toObj < $fromObj) { $tmp = $fromObj; $fromObj = $toObj; $toObj = $tmp; }

$from_dt = $fromObj->format('Y-m-d') . ' 00:00:00';
$to_dt   = $toObj->format('Y-m-d')   . ' 23:59:59';

/* ========== دوال مساعدة ========== */
function http_post_json($url, $token, array $payload, $timeout = 120) {
    $ch = curl_init($url);
    $json = json_encode($payload, JSON_UNESCAPED_UNICODE);
    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_POST           => true,
        CURLOPT_HTTPHEADER     => [
            'Content-Type: application/json',
            'X-ACE-TOKEN: ' . $token
        ],
        CURLOPT_POSTFIELDS     => $json,
        CURLOPT_CONNECTTIMEOUT => 10,
        CURLOPT_TIMEOUT        => $timeout,
    ]);
    $resp = curl_exec($ch);
    $err  = curl_error($ch);
    $code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);

    if ($err)  throw new Exception("cURL error: $err");
    if ($code >= 400) throw new Exception("HTTP $code: $resp");

    $data = json_decode($resp, true);
    if (!is_array($data)) throw new Exception("Invalid JSON response: $resp");
    return $data;
}

/* هل يوجد عمود idcustomer في qrcode؟ (لجعل external_ref ثابتًا) */
$has_idcustomer = false;
$chk = $conn->query("SHOW COLUMNS FROM `qrcode` LIKE 'idcustomer'");
if ($chk && $chk->num_rows > 0) $has_idcustomer = true;

/* تعبير حساب الإجمالي بدون ضريبة (كما في تقريرك) */
$DEFAULT_VAT_RATE_SQL = rtrim(rtrim(number_format((float)$DEFAULT_VAT_RATE, 6, '.', ''), '0'), '.');
$sql_total_without_vat = "
CAST(
  CASE
    WHEN COALESCE(NULLIF(amountwvat,''),'') REGEXP '^[0-9]+(\\.[0-9]+)?$'
      THEN CAST(amountwvat AS DECIMAL(18,4))
    ELSE
      CAST(totalamountwvat AS DECIMAL(18,4)) / (1 + {$DEFAULT_VAT_RATE_SQL})
  END
AS DECIMAL(18,2))";
$sql_total_with_vat = "CAST(totalamountwvat AS DECIMAL(18,2))";

/* الاستعلام الأساسي */
$select_cols = "
      idinvoice,
      customername,
      servicename,
      {$sql_total_without_vat} AS total_without_vat,
      {$sql_total_with_vat}    AS total_with_vat,
      timeadded
";
if ($has_idcustomer) {
    $select_cols = "idcustomer, " . $select_cols;
}

$sql_rows = "
  SELECT
      {$select_cols}
  FROM qrcode
  WHERE COALESCE(LOWER(incomtype),'') = 'confirmed'
    AND timeadded BETWEEN ? AND ?
  ORDER BY timeadded ASC, idinvoice ASC
";

/* ========== القراءة على دفعات والإرسال ========== */
$stmt = $conn->prepare($sql_rows);
$stmt->bind_param('ss', $from_dt, $to_dt);
$stmt->execute();
$res = $stmt->get_result();

$all = [];
while ($r = $res->fetch_assoc()) $all[] = $r;

$total_rows = count($all);
$sent = 0;
$errors = [];

for ($i = 0; $i < $total_rows; $i += $BATCH_SIZE) {
    $slice = array_slice($all, $i, $BATCH_SIZE);

    // بناء partners + invoices
    $partners_map = [];  // key => partner payload (external_ref => obj)
    $invoices     = [];

    foreach ($slice as $row) {
        $idinvoice  = (string)$row['idinvoice'];
        $cust_name  = (string)$row['customername'];
        $service    = (string)$row['servicename'];
        $t_wo_vat   = (float)$row['total_without_vat'];
        $t_w_vat    = (float)$row['total_with_vat'];
        $timeadded  = (string)$row['timeadded'];

        // مفتاح الشريك الخارجي
        if ($has_idcustomer && isset($row['idcustomer']) && $row['idcustomer'] !== '') {
            $external_ref = 'CUST-' . $row['idcustomer'];
        } else {
            // بديل آمن إن لم يتوفر idcustomer
            $external_ref = 'CUSTNAME-' . substr(sha1($cust_name), 0, 12);
        }

        // اجمع الشركاء (دون تكرار)
        if (!isset($partners_map[$external_ref])) {
            $partners_map[$external_ref] = [
                'external_ref' => $external_ref,
                'name'   => $cust_name ?: 'عميل بدون اسم',
                // بإمكانك لاحقًا تمرير phone/mobile/email إن كانت لديك أعمدة
                'phone'  => null,
                'mobile' => null,
                'email'  => null,
                'vat'    => null,
                'is_company' => false,
            ];
        }

        // سطر فاتورة: سعر *قبل* الضريبة + ضريبة 15% (سيتم احتسابها داخل أودو)
        // نرسل أيضًا الإجمالي مع الضريبة لأغراض التحقق إن أردت.
        $invoices[] = [
            'external_id'   => 'INV-' . $idinvoice,     // مفتاح خارجي فريد للفواتير
            'partner_ref'   => $external_ref,           // ربط الفاتورة بالعميل الذي فوق
            'invoice_date'  => substr($timeadded, 0, 10),
            'currency'      => 'SAR',
            'lines' => [[
                'name'        => $service ?: 'خدمة',
                'quantity'    => 1,
                'price_unit'  => round($t_wo_vat, 2),   // قبل الضريبة
                'tax_rate'    => 15.0,                  // % للبحث/التطبيق في أودو
            ]],
            'totals_hint' => [
                'total_wo_vat' => round($t_wo_vat, 2),
                'total_w_vat'  => round($t_w_vat, 2),
            ],
        ];
    }

    $payload = [
        'batch_id' => 'invoices_' . $i . '_' . min($i + $BATCH_SIZE - 1, $total_rows - 1),
        'partners' => array_values($partners_map),
        'invoices' => $invoices,
    ];

    try {
        $resp = http_post_json($ODOO_ENDPOINT, $ODOO_TOKEN, $payload);
        $sent += count($invoices);
        // سجّل نتيجة الدفعة (اختياري)
        @file_put_contents(__DIR__.'/push_to_odoo.log', date('c') . ' ' . $payload['batch_id'] . ' => ' . json_encode($resp, JSON_UNESCAPED_UNICODE) . PHP_EOL, FILE_APPEND);
    } catch (Throwable $e) {
        $errors[] = $e->getMessage();
        @file_put_contents(__DIR__.'/push_to_odoo.log', date('c') . ' ' . $payload['batch_id'] . ' ERROR: ' . $e->getMessage() . PHP_EOL, FILE_APPEND);
    }
}

/* ========== إخراج بسيط في المتصفح ========== */
header('Content-Type: application/json; charset=UTF-8');
echo json_encode([
    'ok'          => empty($errors),
    'from'        => $from_dt,
    'to'          => $to_dt,
    'total_rows'  => $total_rows,
    'sent'        => $sent,
    'batch_size'  => $BATCH_SIZE,
    'errors'      => $errors,
], JSON_UNESCAPED_UNICODE);
